program WMIDemo;
//------------------------------------------------------------------------------
//  File name:      WMIDemo.dpr
//  Last updated:   11/16/03
//  Author:         Sergey Mishkovskiy
//  Company:        USysWare, Inc.
//  Contact info:   usysware@comcast.net
//
//  Compatibility:  Borland Delphi for .NET
//
//  Description:    WMI Demo application.
//------------------------------------------------------------------------------

{$APPTYPE CONSOLE}

{%DelphiDotNetAssemblyCompiler '$(SystemRoot)\microsoft.net\framework\v1.1.4322\System.Management.dll'}
{%DelphiDotNetAssemblyCompiler '$(SystemRoot)\microsoft.net\framework\v1.1.4322\system.drawing.dll'}

uses
  System.Management;

type
  TStringsArray = array of string;

var
  Scope: ManagementScope;

function GetManagementCollection(Scope: ManagementScope;
  const WMIClassName: string): ManagementObjectCollection;
var
  Query: ObjectQuery;
  Searcher: ManagementObjectSearcher;
begin
  // Query the WMI
  Query := ObjectQuery.Create('select * from ' + WMIClassName);
  Searcher := ManagementObjectSearcher.Create(Scope, Query);
  Result := Searcher.Get;
end;

function GetManagementCollectionEx(Scope: ManagementScope;
  const WMIClassName, WhereClause: string): ManagementObjectCollection;
var
  Query: ObjectQuery;
  Searcher: ManagementObjectSearcher;
begin
  // Query the WMI
  Query := ObjectQuery.Create('select * from ' + WMIClassName +
    ' where ' + WhereClause);
  Searcher := ManagementObjectSearcher.Create(Scope, Query);
  Result := Searcher.Get;
end;

function GMTToLocalDateTime(const DT: string): DateTime;
var
  GMT: TimeSpan;
begin
  GMT := TimeZone.CurrentTimeZone.GetUtcOffset(DateTime.Now);

  Result := DateTime.Parse(
    DT.Substring(4, 2) + '/' +
    DT.Substring(6, 2) + '/' +
    DT.Substring(0, 4) + ' ' +
    DT.Substring(8, 2) + ':' +
    DT.Substring(10,2) + ':' +
    DT.Substring(12,2)).Subtract(GMT);
end;

function GMTToLocalDateTimeStr(const DT: string): string;
begin
  Result := GMTToLocalDateTime(DT).ToString;
end;

procedure WMIGetOSInfo(Scope: ManagementScope);
var
  QueryCollection: ManagementObjectCollection;
  Enumerator: ManagementObjectCollection.ManagementObjectEnumerator;
  Item: ManagementObject;
  UpTime: TimeSpan;
  TotalRam, FreeRam, UsedRam: System.Double;
begin
  QueryCollection := GetManagementCollection(Scope, 'Win32_OperatingSystem');
  Enumerator := QueryCollection.GetEnumerator;

  Console.WriteLine;
  Console.WriteLine('OS Information');
  Console.WriteLine('--------------');
  while Enumerator.MoveNext do
  begin
    Item := Enumerator.Current as ManagementObject;
    with Item, Console do
    begin
      UpTime := DateTime.Now.Subtract(
        GMTToLocalDateTime(GetPropertyValue('LastBootupTime').ToString));

      TotalRam := System.Double.Parse(
        GetPropertyValue('TotalVisibleMemorySize').ToString)/1024;
      FreeRam := System.Double.Parse(
        GetPropertyValue('FreePhysicalMemory').ToString)/1024;
      UsedRam := TotalRam - FreeRam;

      WriteLine('Name: ' + GetPropertyValue('Caption').ToString);
      WriteLine('Version: ' + GetPropertyValue('Version').ToString);
      WriteLine('Build: ' + GetPropertyValue('BuildNumber').ToString);
      if UInt16.Parse(
         GetPropertyValue('ServicePackMajorVersion').ToString) <> 0 then
        WriteLine('Service Pack: ' +
        GetPropertyValue('ServicePackMajorVersion').ToString);
      WriteLine('Installed Date: ' +
        GMTToLocalDateTimeStr(GetPropertyValue('InstallDate').ToString));
      WriteLine('Last Rebooted: ' + 
        GMTToLocalDateTimeStr(GetPropertyValue('LastBootupTime').ToString));
      WriteLine('Up Time: ' + UpTime.Days.ToString + ' days, ' +
        UpTime.Hours.ToString + ' hours, ' +
        UpTime.Minutes.ToString + ' minutes');
      WriteLine('Windows Directory: ' +
        GetPropertyValue('WindowsDirectory').ToString);
      WriteLine('System Directory: ' +
        GetPropertyValue('SystemDirectory').ToString);
      WriteLine('RAM Total: ' + Math.Ceiling(TotalRam).ToString + ' MB');
      WriteLine('RAM Used: ' + Math.Ceiling(UsedRam).ToString + ' MB');
      WriteLine('Computer Name: ' + GetPropertyValue('CSName').ToString);
      WriteLine('Number Of Processes: ' +
        GetPropertyValue('NumberOfProcesses').ToString);
    end;
  end;
end;

procedure WMIGetComputerInfo(Scope: ManagementScope);
var
  QueryCollection: ManagementObjectCollection;
  Enumerator: ManagementObjectCollection.ManagementObjectEnumerator;
  Item: ManagementObject;
begin
  QueryCollection := GetManagementCollection(Scope, 'Win32_ComputerSystem');
  Enumerator := QueryCollection.GetEnumerator;

  Console.WriteLine;
  Console.WriteLine('Computer Information');
  Console.WriteLine('--------------------');
  while Enumerator.MoveNext do
  begin
    Item := Enumerator.Current as ManagementObject;
    with Item, Console do
    begin
      WriteLine('Computer Type: ' + GetPropertyValue('SystemType').ToString);
      WriteLine('Manufacturer: ' + GetPropertyValue('Manufacturer').ToString);
      WriteLine('Model: ' + GetPropertyValue('Model').ToString);
      WriteLine('Number Of Processors: ' +
        GetPropertyValue('NumberOfProcessors').ToString);
      WriteLine('Domain: ' + GetPropertyValue('Domain').ToString);
      WriteLine('Logged On User: ' + GetPropertyValue('UserName').ToString);
    end;
  end;
end;

procedure WMIGetBIOSInfo(Scope: ManagementScope);
var
  QueryCollection: ManagementObjectCollection;
  Enumerator: ManagementObjectCollection.ManagementObjectEnumerator;
  Item: ManagementObject;
begin
  QueryCollection := GetManagementCollection(Scope, 'Win32_BIOS');
  Enumerator := QueryCollection.GetEnumerator;

  Console.WriteLine;
  Console.WriteLine('BIOS Information');
  Console.WriteLine('----------------');
  while Enumerator.MoveNext do
  begin
    Item := Enumerator.Current as ManagementObject;
    with Item, Console do
    begin
      WriteLine('BIOS: ' + GetPropertyValue('Version').ToString);
      WriteLine('Manufacturer: ' + GetPropertyValue('Manufacturer').ToString);
      WriteLine('Release Date: ' +
        GMTToLocalDateTimeStr(GetPropertyValue('ReleaseDate').ToString));
      WriteLine('Serial Number: ' + GetPropertyValue('SerialNumber').ToString);
    end;
  end;
end;

procedure WMIGetCDROMInfo(Scope: ManagementScope);
var
  QueryCollection: ManagementObjectCollection;
  Enumerator: ManagementObjectCollection.ManagementObjectEnumerator;
  Item: ManagementObject;
begin
  QueryCollection := GetManagementCollection(Scope, 'Win32_CDROMDrive');
  Enumerator := QueryCollection.GetEnumerator;

  Console.WriteLine;
  Console.WriteLine('CD-ROM Information');
  Console.WriteLine('------------------');
  while Enumerator.MoveNext do
  begin
    Item := Enumerator.Current as ManagementObject;
    with Item, Console do
    begin
      WriteLine('Name: ' + GetPropertyValue('Caption').ToString);
      WriteLine('Description: ' + GetPropertyValue('Description').ToString);
      WriteLine('Drive: ' + GetPropertyValue('Drive').ToString);
    end;
  end;
end;

procedure WMIGetVideoInfo(Scope: ManagementScope);
var
  WinXPandUp: Boolean;
  QueryCollection: ManagementObjectCollection;
  Enumerator: ManagementObjectCollection.ManagementObjectEnumerator;
  Item: ManagementObject;
  Value: System.Double;
begin
  WinXPandUp := ((Environment.OSVersion.Version.Major > 5) or
    ((Environment.OSVersion.Version.Major = 5) and
     (Environment.OSVersion.Version.Minor >= 1)));

  if WinXPandUp then
    QueryCollection := GetManagementCollection(
      Scope, 'Win32_VideoController')
  else
    QueryCollection := GetManagementCollection(
      Scope, 'Win32_VideoConfiguration');
  Enumerator := QueryCollection.GetEnumerator;

  Console.WriteLine;
  Console.WriteLine('Video Information');
  Console.WriteLine('-----------------');
  while Enumerator.MoveNext do
  begin
    Item := Enumerator.Current as ManagementObject;
//    with Item, Console do
    with Console do
    begin
      if WinXPandUp then
      begin
        WriteLine('Card: ' + Item.GetPropertyValue('AdapterCompatibility').ToString);
        if (Item.GetPropertyValue('VideoProcessor') <> nil) then
          WriteLine('Type: ' + Item.GetPropertyValue('VideoProcessor').ToString);
      end
      else
      begin
        WriteLine('Card: ' + Item.GetPropertyValue('AdapterChipType').ToString);
        WriteLine('Type: ' + Item.GetPropertyValue('AdapterType').ToString);
      end;
      if (Item.GetPropertyValue('AdapterRAM')<> nil) then
      begin
        Value := System.Double.Parse(
          Item.GetPropertyValue('AdapterRAM').ToString)/1024/1024;
        WriteLine('RAM: ' + Value.ToString + ' MB');
      end;
    end;
  end;
end;

procedure WMIGetRegistryInfo(Scope: ManagementScope);
var
  QueryCollection: ManagementObjectCollection;
  Enumerator: ManagementObjectCollection.ManagementObjectEnumerator;
  Item: ManagementObject;
begin
  QueryCollection := GetManagementCollection(Scope, 'Win32_Registry');
  Enumerator := QueryCollection.GetEnumerator;

  Console.WriteLine;
  Console.WriteLine('Registry Information');
  Console.WriteLine('--------------------');
  while Enumerator.MoveNext do
  begin
    Item := Enumerator.Current as ManagementObject;
    with Item, Console do
    begin
      WriteLine('Current Size: ' + GetPropertyValue('CurrentSize').ToString +
        ' MB');
      WriteLine('Maximum Size: ' + GetPropertyValue('MaximumSize').ToString +
        ' MB');
    end;
  end;
end;

procedure WMIGetDrivesInfo(Scope: ManagementScope);
var
  QueryCollection: ManagementObjectCollection;
  Enumerator: ManagementObjectCollection.ManagementObjectEnumerator;
  Item: ManagementObject;
  TotalRam, FreeRam, UsedRam: System.Double;
begin
  QueryCollection := GetManagementCollectionEx(Scope, 'Win32_LogicalDisk',
    'DriveType<>2 and DriveType<>5');
  Enumerator := QueryCollection.GetEnumerator;

  Console.WriteLine;
  Console.WriteLine('Non-Removable Drives Information');
  Console.WriteLine('--------------------------------');
  while Enumerator.MoveNext do
  begin
    Item := Enumerator.Current as ManagementObject;
    with Item, Console do
      case uint32.Parse(GetPropertyValue('DriveType').ToString) of
        3:
          begin
            TotalRam := Math.Round(System.Double.Parse(
              GetPropertyValue('Size').ToString)/1024/1024/1024,1);
            FreeRam := Math.Round(System.Double.Parse(
              GetPropertyValue('FreeSpace').ToString)/1024/1024/1024,1);
            UsedRam := TotalRam - FreeRam;

            WriteLine(GetPropertyValue('DeviceID').ToString + ' Used: ' +
              UsedRam.ToString + ' GB Total: ' + TotalRam.ToString + ' GB');
          end;
        4: WriteLine(GetPropertyValue('DeviceID').ToString + ' ' +
          GetPropertyValue('ProviderName').ToString);
      else
        WriteLine(GetPropertyValue('DeviceID').ToString + ' (' +
          GetPropertyValue('Description').ToString + ')');
      end;
  end;
end;

procedure WMIGetNetConfigInfo(Scope: ManagementScope);
var
  QueryCollection: ManagementObjectCollection;
  Enumerator: ManagementObjectCollection.ManagementObjectEnumerator;
  Item: ManagementObject;
  Items: TStringsArray;
  Index: Integer;
begin
  QueryCollection := GetManagementCollection(Scope,
    'Win32_NetworkAdapterConfiguration');
  Enumerator := QueryCollection.GetEnumerator;

  Console.WriteLine;
  Console.WriteLine('Network Configuration Information');
  Console.WriteLine('---------------------------------');
  while Enumerator.MoveNext do
  begin
    Item := Enumerator.Current as ManagementObject;
    with Item, Console do
      if Boolean(GetPropertyValue('IPEnabled')) then
      begin
        WriteLine('NIC: ' + GetPropertyValue('description').ToString);
        WriteLine('MAC Address: ' + GetPropertyValue('MACAddress').ToString);
        if GetPropertyValue('DHCPServer') <> nil then
          WriteLine('DHCP Server: ' + GetPropertyValue('DHCPServer').ToString)
        else
          WriteLine('DHCP Server: ');
        WriteLine('DHCP Enabled: ' + GetPropertyValue('DHCPenabled').ToString);
        if GetPropertyValue('DNSDomain') <> nil then
          WriteLine('DNS Domain: ' + GetPropertyValue('DNSDomain').ToString)
        else
          WriteLine('DNS Domain: ');
        WriteLine('DNS Host: ' + GetPropertyValue('DNSHostName').ToString);

        Items := TStringsArray(GetPropertyValue('IPAddress'));
        if Items <> nil then
        begin
          WriteLine('IP Addresses:');
          for Index := 0 to Length(Items) - 1 do
            WriteLine('  ' + Items[Index]);
        end;

        Items := TStringsArray(GetPropertyValue('IPSubnet'));
        if Items <> nil then
        begin
          WriteLine('Subnet Masks:');
          for Index := 0 to Length(Items) - 1 do
            WriteLine('  ' + Items[Index]);
        end;

        Items := TStringsArray(GetPropertyValue('DefaultIPGateway'));
        if Items <> nil then
        begin
          WriteLine('Default Gateways:');
          for Index := 0 to Length(Items) - 1 do
            WriteLine('  ' + Items[Index]);
        end;

        Items := TStringsArray(GetPropertyValue('DNSServerSearchOrder'));
        if Items <> nil then
        begin
          WriteLine('DNS Servers:');
          for Index := 0 to Length(Items) - 1 do
            WriteLine('  ' + Items[Index]);
        end;
      end;
  end;
end;

procedure WMIGetSharesInfo(Scope: ManagementScope);
var
  QueryCollection: ManagementObjectCollection;
  Enumerator: ManagementObjectCollection.ManagementObjectEnumerator;
  Item: ManagementObject;
begin
  QueryCollection := GetManagementCollection(Scope, 'Win32_Share');
  Enumerator := QueryCollection.GetEnumerator;

  Console.WriteLine;
  Console.WriteLine('Shares Information');
  Console.WriteLine('------------------');
  while Enumerator.MoveNext do
  begin
    Item := Enumerator.Current as ManagementObject;
    with Item, Console do
      WriteLine(GetPropertyValue('Name').ToString + ' - ' +
        GetPropertyValue('Path').ToString);
  end;
end;

procedure WMIGetActiveServicesInfo(Scope: ManagementScope);
var
  QueryCollection: ManagementObjectCollection;
  Enumerator: ManagementObjectCollection.ManagementObjectEnumerator;
  Item: ManagementObject;
begin
  QueryCollection := GetManagementCollectionEx(Scope, 'Win32_Service',
    'State=''Running''');
  Enumerator := QueryCollection.GetEnumerator;

  Console.WriteLine;
  Console.WriteLine('Active Services Information');
  Console.WriteLine('---------------------------');
  while Enumerator.MoveNext do
  begin
    Item := Enumerator.Current as ManagementObject;
    with Item, Console do
      WriteLine(GetPropertyValue('DisplayName').ToString + ' - ' +
        GetPropertyValue('StartMode').ToString);
  end;
end;

procedure WMIGetPrintersInfo(Scope: ManagementScope);
var
  QueryCollection: ManagementObjectCollection;
  Enumerator: ManagementObjectCollection.ManagementObjectEnumerator;
  Item: ManagementObject;
begin
  QueryCollection := GetManagementCollection(Scope, 'Win32_Printer');
  Enumerator := QueryCollection.GetEnumerator;

  Console.WriteLine;
  Console.WriteLine('Printers Information');
  Console.WriteLine('--------------------');
  while Enumerator.MoveNext do
  begin
    Item := Enumerator.Current as ManagementObject;
    with Item, Console do
    begin
      Console.Write(GetPropertyValue('DeviceID').ToString + ' - ');
      case uint16.Parse(GetPropertyValue('PrinterStatus').ToString) of
        3: Console.Write('Idle');
        4: Console.Write('Printing');
        5: Console.Write('Warmup');
        6: Console.Write('Stopped Printing');
        7: Console.Write('Offline');
      else
        Console.Write('Unknown');
      end;
      Console.WriteLine;
    end;
  end;
end;

procedure WMIGetStartupInfo(Scope: ManagementScope);
var
  QueryCollection: ManagementObjectCollection;
  Enumerator: ManagementObjectCollection.ManagementObjectEnumerator;
  Item: ManagementObject;
begin
  QueryCollection := GetManagementCollection(Scope, 'Win32_StartupCommand');
  Enumerator := QueryCollection.GetEnumerator;

  Console.WriteLine;
  Console.WriteLine('Startup Programs Information');
  Console.WriteLine('----------------------------');
  while Enumerator.MoveNext do
  begin
    Item := Enumerator.Current as ManagementObject;
    with Item, Console do
      WriteLine(GetPropertyValue('Name').ToString + ' - ' +
        GetPropertyValue('Command').ToString);
  end;
end;

procedure WMIGetEventLogInfo(Scope: ManagementScope);
var
  QueryCollection: ManagementObjectCollection;
  Enumerator: ManagementObjectCollection.ManagementObjectEnumerator;
  Item: ManagementObject;
  Count: Integer;
begin
  QueryCollection := GetManagementCollectionEx(Scope, 'Win32_NTLogEvent',
    'EventType=1');
  Enumerator := QueryCollection.GetEnumerator;
  Count := 0;

  Console.WriteLine;
  Console.WriteLine('Application EventLog last 10 errors Information');
  Console.WriteLine('-----------------------------------------------');
  while Enumerator.MoveNext do
  begin
    Item := Enumerator.Current as ManagementObject;
    with Item, Console do
    begin
      Inc(Count);

      Console.Write(GetPropertyValue('EventCode').ToString + ' - ' +
        GMTToLocalDateTimeStr(GetPropertyValue('TimeGenerated').ToString));
      if GetPropertyValue('Message') <> nil then
      begin
        Console.Write(' - ');
        if GetPropertyValue('Message').ToString.Length > 40 then
          Console.Write(
            GetPropertyValue('Message').ToString.Substring(0, 40) + '...')
        else
          Console.Write(GetPropertyValue('Message').ToString);
      end;
      Console.WriteLine;

      if Count > 10 then
        Break;
    end;
  end;
end;

begin
  Console.WriteLine;
  Console.WriteLine('WMI Demo');
  Console.WriteLine('--------');

  Scope := System.Management.ManagementScope.Create('\\localhost\root\cimv2');

  WMIGetOSInfo(Scope);
  WMIGetComputerInfo(Scope);
  WMIGetBIOSInfo(Scope);
  WMIGetCDROMInfo(Scope);
  WMIGetVideoInfo(Scope);
  WMIGetRegistryInfo(Scope);
  WMIGetDrivesInfo(Scope);
  WMIGetNetConfigInfo(Scope);
  WMIGetSharesInfo(Scope);
  WMIGetPrintersInfo(Scope);
  WMIGetStartupInfo(Scope);
  WMIGetActiveServicesInfo(Scope);
  WMIGetEventLogInfo(Scope);

  Console.WriteLine;
  Console.WriteLine('Done. Press <Enter> to exit...');
  Console.WriteLine('------------------------------');
  Console.ReadLine;
end.
